<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://ays-pro.com/
 * @since      1.0.0
 *
 * @package    Chart_Builder
 * @subpackage Chart_Builder/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Chart_Builder
 * @subpackage Chart_Builder/admin
 * @author     Chart Builder Team <info@ays-pro.com>
 */
class Chart_Builder_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The capability of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $capability    The capability for users access to this plugin.
	 */
    private $capability;

	/**
	 * @var Chart_Builder_DB_Actions
	 */
	private $db_obj;

	/**
	 * @var Chart_Builder_Settings_DB_Actions
	 */
	private $settings_obj;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

        add_filter('set-screen-option', array(__CLASS__, 'set_screen'), 10, 3);
		add_filter('set_screen_option_cb_charts_per_page', array(__CLASS__, 'set_screen'), 10, 3);

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles($hook_suffix) {
		wp_enqueue_style( $this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'css/admin.css', array(), $this->version, 'all');
        // Not enqueue styles if they are not on the current plugin page
        if (false === strpos($hook_suffix, $this->plugin_name)) return;
        //
		wp_enqueue_style( $this->plugin_name . '-normalize', plugin_dir_url( __FILE__ ) . 'css/normalize.css', array(), $this->version . time(), 'all' );
		wp_enqueue_style( $this->plugin_name . '-admin-general', plugin_dir_url( __FILE__ ) . 'css/admin-general.css', array(), $this->version, 'all' );
		wp_enqueue_style( $this->plugin_name . '-banner', plugin_dir_url( __FILE__ ) . 'css/banner.css', array(), $this->version, 'all' );
		wp_enqueue_style( $this->plugin_name . '-chart-builder-banner.css', plugin_dir_url(__FILE__) . 'css/chart-builder-banner.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-animate', plugin_dir_url(__FILE__) .  'css/animate.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-font-awesome', plugin_dir_url(__FILE__) .  'css/ays-font-awesome.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-font-awesome-icons', plugin_dir_url(__FILE__) .  'css/ays-font-awesome-icons.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-select2', plugin_dir_url(__FILE__) .  'css/ays-select2.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-chosen', plugin_dir_url(__FILE__) .  'css/chosen.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-bootstrap', plugin_dir_url(__FILE__) . 'css/bootstrap.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-data-bootstrap', plugin_dir_url(__FILE__) . 'css/dataTables.bootstrap4.min.css', array(), $this->version, 'all');

		wp_enqueue_style( $this->plugin_name . '-layer', plugin_dir_url( __FILE__ ) . 'css/chart-builder-admin-layer.css', array(), $this->version, 'all' );
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/chart-builder-admin.css', array(), $this->version . time(), 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts($hook_suffix) {
		if (false !== strpos($hook_suffix, "plugins.php")){
            wp_enqueue_script($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'js/admin.js', array('jquery'), $this->version, true);
			wp_localize_script($this->plugin_name . '-admin',  'chart_builder_admin_ajax', array('ajax_url' => admin_url('admin-ajax.php')));
        }

        // Not enqueue scripts if they are not on the current plugin page
        if (false === strpos($hook_suffix, $this->plugin_name)) return;
        //
        global $wp_version;
        $version1 = $wp_version;
        $operator = '>=';
        $version2 = '5.5';
        $versionCompare = CBFunctions()->versionCompare( $version1, $operator, $version2 );

        if ( $versionCompare ) {
            wp_enqueue_script( $this->plugin_name.'-wp-load-scripts', plugin_dir_url(__FILE__) . 'js/load-scripts.js', array(), $this->version, true);
        }

        wp_enqueue_script( 'jquery' );

        /*
        ==========================================
           * Bootstrap
           * select2
           * jQuery DataTables
        ==========================================
        */
        wp_enqueue_script( $this->plugin_name . "-popper", plugin_dir_url(__FILE__) . 'js/popper.min.js', array( 'jquery' ), $this->version, true );
        wp_enqueue_script( $this->plugin_name . "-bootstrap", plugin_dir_url(__FILE__) . 'js/bootstrap.min.js', array( 'jquery' ), $this->version, true );
        wp_enqueue_script( $this->plugin_name . '-select2js', plugin_dir_url( __FILE__ ) . 'js/ays-select2.min.js', array('jquery'), $this->version, true);
        wp_enqueue_script( $this->plugin_name . '-chosen', plugin_dir_url( __FILE__ ) . 'js/chosen.jquery.min.js', array('jquery'), $this->version, true);
        wp_enqueue_script( $this->plugin_name . '-datatable-min', plugin_dir_url( __FILE__ ) . 'js/chart-builder-datatable.min.js', array('jquery'), $this->version, true);
        wp_enqueue_script( $this->plugin_name . "-db4.min.js", plugin_dir_url( __FILE__ ) . 'js/dataTables.bootstrap4.min.js', array( 'jquery' ), $this->version, true );

		$table_col_mapping  = CBFunctions()->get_all_db_tables_column_mapping( 1 );
        $chart_banner_date = $this->ays_chart_update_banner_time();

		wp_enqueue_code_editor(
			array(
				'type' => 'sql',
				'codemirror' => array(
					'autofocus'         => true,
					'lineWrapping'      => true,
					'dragDrop'          => false,
					'matchBrackets'     => true,
					'autoCloseBrackets' => true,
					'extraKeys'         => array( 'Ctrl-Space' => 'autocomplete' ),
					'hintOptions'       => array( 'tables' => $table_col_mapping ),
				),
			)
		);

        wp_enqueue_script( $this->plugin_name . '-charts-google', plugin_dir_url(__FILE__) . 'js/google-chart.js', array('jquery'), $this->version, true);

		wp_enqueue_script( $this->plugin_name . '-functions', plugin_dir_url( __FILE__ ) . 'js/functions.js', array( 'jquery' ), $this->version, true );
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/chart-builder-admin.js', array( 'jquery' ), $this->version .time(), true );

		wp_enqueue_script( $this->plugin_name . "-general-js", plugin_dir_url( __FILE__ ) . 'js/chart-builder-admin-general.js', array( 'jquery' ), $this->version, true );
		wp_localize_script( $this->plugin_name, 'aysChartBuilderAdmin', array(
			'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'selectUser'                        => __( 'Select user', "chart-builder" ),
            'pleaseEnterMore'                   => __( "Please enter 1 or more characters", "chart-builder" ),
            'searching'                         => __( "Searching...", "chart-builder" ),
            'chartBannerDate'                   => $chart_banner_date,
			'selectUserRoles'                   => __( 'Select user roles', "chart-builder" ),
			'delete'                            => __( 'Delete', "chart-builder" ),
			'selectQuestionDefaultType'         => __( 'Select question default type', "chart-builder" ),
			'yes'                               => __( 'Yes', "chart-builder" ),
			'cancel'                            => __( 'Cancel', "chart-builder" ),
			'somethingWentWrong'                => __( "Maybe something went wrong.", "chart-builder" ),
			'failed'                            => __( 'Failed', "chart-builder" ),
			'selectPage'                        => __( 'Select page', "chart-builder" ),
			'selectPostType'                    => __( 'Select post type', "chart-builder" ),
			'copied'                            => __( 'Copied!', "chart-builder"),
			'clickForCopy'                      => __( 'Click to copy', "chart-builder"),
			'selectForm'                        => __( 'Select form', "chart-builder"),
			'confirmDelete'                     => __( 'Are you sure you want to delete the chart(s)?', "chart-builder"),
			'confirmRowDelete'                  => __( 'Are you sure you want to delete the row?', "chart-builder"),
			'confirmColDelete'                  => __( 'Are you sure you want to delete the column?', "chart-builder"),
			'minRowNotice'                      => __( 'Sorry, minimum count of rows should be 1', "chart-builder"),
			'minColNotice'                      => __( 'Sorry, minimum count of columns should be 1', "chart-builder"),            
		) );

		wp_localize_script(
			$this->plugin_name, 'aysChartBuilderChartSettings', array(
				'types' => CBFunctions()->getAllowedTypes(),
				'max_selected_options' => 2,
				'l10n' => array(
					'invalid_source'      => esc_html__( 'You have entered invalid URL. Please, insert proper URL.', "chart-builder" ),
					'loading'             => esc_html__( 'Loading...', "chart-builder" ),
					'filter_config_error' => esc_html__( 'Please check the filters you have configured.', "chart-builder" ),
					'select_columns'      => esc_html__( 'Please select a few columns to include in the chart.', "chart-builder" ),
					'save_settings'       => __( 'You have modified the chart\'s settings. To modify the source/data again, you must save this chart and reopen it for editing. If you continue without saving the chart, you may lose your changes.', "chart-builder" ),
				),
				'ajax' => array(
					'url' => admin_url( 'admin-ajax.php' ),
					'nonces' => array(
						'filter_get_props' => wp_create_nonce( 'cbuilder-fetch-post-type-props' ),
						'filter_get_data'  => wp_create_nonce( 'cbuilder-fetch-post-type-data' ),
                        'quiz_maker_get_data' => wp_create_nonce( 'cbuilder-fetch-quiz-maker-data' ),
						'quiz_maker_save_data' => wp_create_nonce( 'cbuilder-save-quiz-maker-data' ),
                        'author_user_search' => wp_create_nonce( 'cbuilder-author-user-search' ),
					),
					'actions' => array(
						'filter_get_props' => 'fetch_post_type_props',
						'filter_get_data' => 'fetch_post_type_data',
                        'quiz_maker_get_data'   => 'fetch_quiz_maker_data',
						'quiz_maker_save_data'   => 'save_quiz_maker_data',
                        'author_user_search' => 'author_user_search'
					),
				),
				'db_query' => array(
					'tables' => $table_col_mapping,
				),
			)
		);

		if ( false !== strpos( $hook_suffix, 'settings' ) ) {
			wp_enqueue_script( $this->plugin_name . '-settings', plugin_dir_url( __FILE__ ) . 'js/chart-builder-admin-settings.js', array( 'jquery' ), $this->version, true );

			wp_localize_script( $this->plugin_name . '-settings', 'aysChartBuilderAdminSettings', array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'selectUserRoles'                   => __( 'Select user roles', $this->plugin_name ),
				'delete'                            => __( 'Delete', $this->plugin_name ),
				'selectQuestionDefaultType'         => __( 'Select question default type', $this->plugin_name ),
				'yes'                               => __( 'Yes', $this->plugin_name ),
				'cancel'                            => __( 'Cancel', $this->plugin_name ),
				'somethingWentWrong'                => __( "Maybe something went wrong.", $this->plugin_name ),
				'failed'                            => __( 'Failed', $this->plugin_name ),
				'selectPage'                        => __( 'Select page', $this->plugin_name ),
				'selectPostType'                    => __( 'Select post type', $this->plugin_name ),
				'copied'                            => __( 'Copied!', $this->plugin_name),
				'clickForCopy'                      => __( 'Click to copy', $this->plugin_name),
				'selectForm'                        => __( 'Select form', $this->plugin_name),
			) );
		}
	}


    /**
     * Register the administration menu for this plugin into the WordPress Dashboard menu.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu(){

        /*
         * Add a settings page for this plugin to the Settings menu.
         *
         * NOTE:  Alternative menu locations are available via WordPress administration menu functions.
         *
         *        Administration Menus: http://codex.wordpress.org/Administration_Menus
         *
         */
        global $wpdb;
        // $sql = "SELECT COUNT(*) FROM " . esc_sql( $wpdb->prefix . EXAM_MAKER_DB_PREFIX ) . "submissions WHERE `read` = 0 OR `read` = 2 ";
        // $unread_results_count = intval( $wpdb->get_var( $sql ) );
        $menu_item = __( 'Chart Builder', "chart-builder" );// : 'Exam Maker' . '<span class="ays-survey-menu-badge ays-survey-results-bage">' . $unread_results_count . '</span>';

        $this->capability = 'manage_options';// $this->survey_maker_capabilities();
        // $this->current_user_can_edit = Survey_Maker_Data::survey_maker_capabilities_for_editing();

        $hook_page_view = add_menu_page(
            __( 'Charts', "chart-builder" ),
            $menu_item,
            $this->capability,
            $this->plugin_name,
            array($this, 'display_plugin_charts_page'),
            CHART_BUILDER_ADMIN_URL . '/images/icons/ays_chart_logo_icon.png',
            '6.22'
        );

	    add_action( "load-$hook_page_view", array( $this, 'screen_option_charts' ) );

    }

    public function add_plugin_dashboard_submenu(){
        $hook_charts = add_submenu_page(
            $this->plugin_name,
            __('How to use', $this->plugin_name),
            __('How to use', $this->plugin_name),
            'manage_options',
            $this->plugin_name . '-dashboard',
            array($this, 'display_plugin_setup_page')
        );
        add_action("load-$hook_charts", array( $this, 'add_tabs' ));
    }

	public function add_plugin_general_settings_submenu(){
		$hook_settings = add_submenu_page( $this->plugin_name,
			__('General Settings', $this->plugin_name),
			__('General Settings', $this->plugin_name),
			'manage_options',
			$this->plugin_name . '-settings',
			array($this, 'display_plugin_settings_page')
		);
		add_action("load-$hook_settings", array($this, 'screen_option_settings'));
	}

    public function add_plugin_featured_plugins_submenu(){
        $hook_our_products = add_submenu_page( $this->plugin_name,
            __('Our products', $this->plugin_name),
            __('Our products', $this->plugin_name),
            'manage_options',
            $this->plugin_name . '-featured-plugins',
            array($this, 'display_plugin_featured_plugins_page') 
        );

        add_action("load-$hook_our_products", array( $this, 'add_tabs' ));
    }

    public function add_plugin_chart_features_submenu(){
        $hook_pro_features = add_submenu_page(
            $this->plugin_name,
            __('PRO Features', $this->plugin_name),
            __('PRO Features', $this->plugin_name),
            'manage_options',
            $this->plugin_name . '-chart-features',
            array($this, 'display_plugin_chart_features_page')
        );

        add_action("load-$hook_pro_features", array( $this, 'add_tabs' ));
    }

	public function screen_option_charts(){
		$option = 'per_page';
		$args = array(
			'label' => __('Charts', "chart-builder"),
			'default' => 5,
			'option' => 'cb_charts_per_page'
		);

		if( ! ( isset( $_GET['action'] ) && ( $_GET['action'] == 'add' || $_GET['action'] == 'edit' ) ) ){
			add_screen_option( $option, $args );
		}

	//		$this->charts_obj = new Surveys_List_Table($this->plugin_name);
		$this->db_obj = new Chart_Builder_DB_Actions( $this->plugin_name );
	//        var_dump($this);
	//		$this->settings_obj = new Survey_Maker_Settings_Actions($this->plugin_name);
	}

	public function screen_option_settings(){
		$this->settings_obj = new Chart_Builder_Settings_DB_Actions( $this->plugin_name );
	}

    public function add_tabs() {
		$screen = get_current_screen();
	
		if ( ! $screen) {
			return;
		}
        
        // $title   = __( 'General Information:', $this->plugin_name);
        // $content_text = 'Get real-time feedback with the Survey Maker plugin. You are free to generate unlimited online surveys with unlimited questions and sections. Easily create your customer satisfaction surveys, employee engagement forms, market researches, event planning questionnaires with this plugin.
        //                 <br><br>Increase users’ track to your WordPress website with the Survey Maker features. Build smarter surveys with LogicJump, advance your questionnaires with Conditional Results, earn money with Paid Surveys, generate leads super easily, get valuable feedback.';

        // $sidebar_content = '<p><strong>' . __( 'For more information:', $this->plugin_name) . '</strong></p>' .
        //                     '<p>
        //                         <a href="https://www.youtube.com/watch?v=Q1qi649acb0" target="_blank">' . __( 'Youtube video tutorials' , $this->plugin_name ) . '</a>
        //                     </p>' .
        //                     '<p>
        //                         <a href="https://ays-pro.com/wordpress-survey-maker-user-manual" target="_blank">' . __( 'Documentation', $this->plugin_name ) . '</a>
        //                     </p>' .
        //                     '<p>
        //                         <a href="https://ays-pro.com/wordpress/survey-maker" target="_blank">' . __( 'Survey Maker plugin pro version', $this->plugin_name ) . '</a>
        //                     </p>' .
        //                     '<p>
        //                         <a href="https://ays-demo.com/wordpress-survey-plugin-pro-demo/" target="_blank">' . __( 'Survey Maker plugin demo', $this->plugin_name ) . '</a>
        //                     </p>';

        
        // $content =  '<h2>' . __( 'Survey Maker Information', $this->plugin_name) . '</h2>'
        //            .'<p>' .sprintf(__( '%s',  $this->plugin_name ), $content_text).'</p>';

        // $help_tab_content = array(
        //     'id'      => 'survey_maker_help_tab',
        //     'title'   => $title,
        //     'content' => $content
        // );
        
		// $screen->add_help_tab($help_tab_content);

		// $screen->set_help_sidebar($sidebar_content);
	}

	public function display_plugin_charts_page(){
        global $ays_chart_db_actions;

        $action = (isset($_GET['action'])) ? sanitize_text_field( $_GET['action'] ) : '';
		$id = (isset($_GET['id'])) ? absint( esc_attr($_GET['id']) ) : 0;

        if (isset($_POST['bulk_delete_confirm'])) {
            if (isset($_POST['bulk-delete']) && !empty($_POST['bulk-delete'])) {
                $ids = $_POST['bulk-delete'];
                foreach ($ids as $id) {
                    if ($id > 0) {
                        $this->db_obj->delete_item( $id );
                    }
                }
                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
	            $url = esc_url_raw( add_query_arg( array(
		            "status" => 'all-deleted'
	            ), $url ) );
	            wp_redirect( $url );
                exit;
            }
        }

        switch ($action) {
            case 'trash':
                if( $id > 0 ){
                    $this->db_obj->trash_item( $id );
	                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
	                $url = esc_url_raw( add_query_arg( array(
		                "status" => 'trashed'
	                ), $url ) );
	                wp_redirect( $url );
	                exit;
                }
                break;
            case 'restore':
                if( $id > 0 ){
                    $this->db_obj->restore_item( $id );
	                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
	                $url = esc_url_raw( add_query_arg( array(
		                "status" => 'restored'
	                ), $url ) );
	                wp_redirect( $url );
	                exit;
                }
                break;
            case 'delete':
                if( $id > 0 ){
                    $this->db_obj->delete_item( $id );
	                $url = remove_query_arg( array('action', 'id', '_wpnonce') );
	                $url = esc_url_raw( add_query_arg( array(
		                "status" => 'deleted'
	                ), $url ) );
	                wp_redirect( $url );
	                exit;
                }
                break;
            case 'add':
                include_once('partials/charts/actions/chart-builder-charts-actions.php');
                break;
            case 'edit':
                include_once('partials/charts/actions/chart-builder-charts-actions.php');
                break;
            default:
                include_once('partials/charts/chart-builder-charts-display.php');
        }
    }

    public function display_plugin_setup_page(){
        include_once('partials/chart-builder-admin-display.php');
    }

	public function display_plugin_settings_page(){
		include_once('partials/settings/chart-builder-settings.php');
	}

    public function display_plugin_featured_plugins_page(){
        include_once('partials/features/chart-builder-plugin-featured-display.php');
    }

    public function display_plugin_chart_features_page(){
        include_once('partials/features/chart-builder-features-display.php');
    }

    /**
     * Add settings action link to the plugins page.
     *
     * @since    1.0.0
     */
    public function add_action_links( $links ){
        /*
        *  Documentation : https://codex.wordpress.org/Plugin_API/Filter_Reference/plugin_action_links_(plugin_file_name)
        */
        $settings_link = array(
            '<a href="' . admin_url('admin.php?page=' . $this->plugin_name) . '">' . __('Settings', "chart-builder") . '</a>',
            '<a href="https://ays-demo.com/chart-builder-demo/" target="_blank">' . __('Demo', "chart-builder") . '</a>',
            '<a href="https://ays-pro.com/wordpress/chart-builder" style="font-weight:bold;color:#93003C;" target="_blank">' . __('Go Pro', "chart-builder") . '</a>',
        );
        return array_merge( $settings_link, $links );

    }


    public static function set_screen($status, $option, $value){
        return $value;
    }

	public function ays_admin_ajax(){
		global $wpdb;

		$response = array(
			"status" => false
		);

		$function = isset($_REQUEST['function']) ? sanitize_text_field( $_REQUEST['function'] ) : null;

		if($function !== null){
			$response = array();
			if( is_callable( array( $this, $function ) ) ){
				$response = $this->$function();

	            ob_end_clean();
	            $ob_get_clean = ob_get_clean();
				echo json_encode( $response );
				wp_die();
			}

        }

        ob_end_clean();
        $ob_get_clean = ob_get_clean();
		echo json_encode( $response );
		wp_die();
	}

    public function deactivate_plugin_option(){
        $request_value = esc_attr($_REQUEST['upgrade_plugin']);
        $upgrade_option = get_option( 'ays_chart_builder_upgrade_plugin', '' );
        if($upgrade_option === ''){
            add_option( 'ays_chart_builder_upgrade_plugin', $request_value );
        }else{
            update_option( 'ays_chart_builder_upgrade_plugin', $request_value );
        }
		ob_end_clean();
        $ob_get_clean = ob_get_clean();
        return json_encode( array( 'option' => get_option( 'ays_chart_builder_upgrade_plugin', '' ) ) );
		wp_die();
    }

    public function chart_builder_admin_footer($a){
        if(isset($_REQUEST['page'])){
            if(false !== strpos( sanitize_text_field( $_REQUEST['page'] ), $this->plugin_name)){
                ?>
                <p style="font-size:13px;text-align:center;font-style:italic;">
                    <span style="margin-left:0px;margin-right:10px;" class="ays_heart_beat"><i class="ays_fa ays_fa_heart_o animated"></i></span>
                    <span><?php echo esc_html(__( "If you love our plugin, please do big favor and rate us on", "chart-builder")); ?></span>
                    <a target="_blank" href='https://wordpress.org/support/plugin/chart-builder/reviews/'>WordPress.org</a>
                    <span class="ays_heart_beat"><i class="ays_fa ays_fa_heart_o animated"></i></span>
                </p>
            <?php
            }
        }
    }

    // Chart Builder Elementor widget init
    public function chart_builder_el_widgets_registered() {
        // We check if the Elementor plugin has been installed / activated.
        if ( defined( 'ELEMENTOR_PATH' ) && class_exists( 'Elementor\Widget_Base' ) ) {
            // get our own widgets up and running:
            // copied from widgets-manager.php
            if ( class_exists( 'Elementor\Plugin' ) ) {
                if ( is_callable( 'Elementor\Plugin', 'instance' ) ) {
                    $elementor = Elementor\Plugin::instance();
                    if ( isset( $elementor->widgets_manager ) ) {
                        if ( method_exists( $elementor->widgets_manager, 'register_widget_type' ) ) {
                            wp_enqueue_style($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'css/admin.css', array(), $this->version, 'all');
                            $widget_file   = 'plugins/elementor/chart-builder-elementor.php';
                            $template_file = locate_template( $widget_file );
                            if ( !$template_file || !is_readable( $template_file ) ) {
                                $template_file = CHART_BUILDER_DIR . 'pb_templates/chart-builder-elementor.php';
                            }
                            if ( $template_file && is_readable( $template_file ) ) {
                                require_once $template_file;
                                Elementor\Plugin::instance()->widgets_manager->register_widget_type( new Elementor\Widget_Chart_Builder_Elementor() );
                            }
                        }
                    }
                }
            }
        }
    }

    public function fetch_post_type_props(){
	    $nonce = isset( $_POST['nonce'] ) ? wp_verify_nonce( sanitize_text_field( $_POST['nonce'] ), 'cbuilder-fetch-post-type-props' ) : '';
	    if ( $nonce ) {
            $results = CBFunctions()->get_post_type_properties( sanitize_text_field( $_POST['post_type'] ) );

		    return array(
			    'success' => true,
			    'fields'  => $results,
		    );
	    }

	    return array(
		    'success' => false,
	    );
    }

	public static function get_max_id( $table ) {
        global $wpdb;
        $db_table = $wpdb->prefix . CHART_BUILDER_DB_PREFIX . $table;

        $sql = "SELECT MAX(id) FROM {$db_table}";

        $result = intval( $wpdb->get_var( $sql ) );

        return $result;
    }

    public static function ays_restriction_string($type, $x, $length){
        $output = "";
        switch($type){
            case "char":                
                if(strlen($x)<=$length){
                    $output = $x;
                } else {
                    $output = substr($x,0,$length) . '...';
                }
                break;
            case "word":
                $res = explode(" ", $x);
                if(count($res)<=$length){
                    $output = implode(" ",$res);
                } else {
                    $res = array_slice($res,0,$length);
                    $output = implode(" ",$res) . '...';
                }
            break;
        }
        return $output;
    }

	public function ays_chart_sale_banner(){
        if(isset($_POST['ays_chart_sale_btn'])){
            update_option('ays_chart_sale_btn', 1);
            update_option('ays_chart_sale_date', current_time( 'mysql' ));
        }

        $ays_chart_sale_date = get_option('ays_chart_sale_date');

        $val = 60*60*24*5;

        $current_date = current_time( 'mysql' );
        $date_diff = strtotime($current_date) - intval(strtotime($ays_chart_sale_date)) ;
        
        $days_diff = $date_diff / $val;
    
        if(intval($days_diff) > 0 ){
            update_option('ays_chart_sale_btn', 0);
        }
    
        $ays_chart_builder_flag = intval(get_option('ays_chart_sale_btn'));
        if( $ays_chart_builder_flag == 0 ){
            if (isset($_GET['page']) && strpos($_GET['page'], CHART_BUILDER_NAME) !== false) {
                if( !(Chart_Builder_Admin::get_max_id('charts') <= 1) ){
                    // $this->ays_chart_sale_message($ays_chart_builder_flag);
                    // $this->ays_chart_spring_bundle_small_message($ays_chart_builder_flag);
                    // $this->ays_chart_builder_helloween_message($ays_chart_builder_flag);
                    // $this->ays_chart_black_friday_message($ays_chart_builder_flag);
                    // $this->ays_chart_christmas_message($ays_chart_builder_flag);
                    $this->ays_chart_silver_bundle_message($ays_chart_builder_flag);
                }
            }
        }
    }

	// Black Friday
    public static function ays_chart_black_friday_message($ishmar){
        if($ishmar == 0 ){
            $content = array();

            $content[] = '<div id="ays-chart-dicount-black-friday-month-main" class="notice notice-success is-dismissible ays_chart_dicount_info">';
                $content[] = '<div id="ays-chart-dicount-black-friday-month" class="ays_chart_dicount_month">';
                    $content[] = '<div class="ays-chart-dicount-black-friday-box">';
                        $content[] = '<div class="ays-chart-dicount-black-friday-wrap-box ays-chart-dicount-black-friday-wrap-box-80" style="width: 70%;">';
                            $content[] = '<div class="ays-chart-dicount-black-friday-title-row">' . __( 'Limited Time', CHART_BUILDER_NAME ) .' '. '<a href="https://ays-pro.com/wordpress/chart-builder" class="ays-chart-dicount-black-friday-button-sale" target="_blank">' . __( 'Sale', CHART_BUILDER_NAME ) . '</a>' . '</div>';
                            $content[] = '<div class="ays-chart-dicount-black-friday-title-row">' . __( 'Chart Builder plugin', CHART_BUILDER_NAME ) . '</div>';
                        $content[] = '</div>';

                        $content[] = '<div class="ays-chart-dicount-black-friday-wrap-box ays-chart-dicount-black-friday-wrap-text-box">';
                            $content[] = '<div class="ays-chart-dicount-black-friday-text-row">' . __( '20% off', CHART_BUILDER_NAME ) . '</div>';
                        $content[] = '</div>';

                        $content[] = '<div class="ays-chart-dicount-black-friday-wrap-box" style="width: 25%;">';
                            $content[] = '<div id="ays-chart-builder-countdown-main-container">';
                                $content[] = '<div class="ays-chart-countdown-container">';
                                    $content[] = '<div id="ays-chart-countdown" style="display: block;">';
                                        $content[] = '<ul>';
                                            $content[] = '<li><span id="ays-chart-countdown-days">0</span>' . __( 'Days', CHART_BUILDER_NAME ) . '</li>';
                                            $content[] = '<li><span id="ays-chart-countdown-hours">0</span>' . __( 'Hours', CHART_BUILDER_NAME ) . '</li>';
                                            $content[] = '<li><span id="ays-chart-countdown-minutes">0</span>' . __( 'Minutes', CHART_BUILDER_NAME ) . '</li>';
                                            $content[] = '<li><span id="ays-chart-countdown-seconds">0</span>' . __( 'Seconds', CHART_BUILDER_NAME ) . '</li>';
                                        $content[] = '</ul>';
                                    $content[] = '</div>';
                                    $content[] = '<div id="ays-chart-countdown-content" class="emoji" style="display: none;">';
                                        $content[] = '<span>🚀</span>';
                                        $content[] = '<span>⌛</span>';
                                        $content[] = '<span>🔥</span>';
                                        $content[] = '<span>💣</span>';
                                    $content[] = '</div>';
                                $content[] = '</div>';
                            $content[] = '</div>';
                        $content[] = '</div>';

                        $content[] = '<div class="ays-chart-dicount-black-friday-wrap-box" style="width: 25%;">';
                            $content[] = '<a href="https://ays-pro.com/wordpress/chart-builder" class="ays-chart-dicount-black-friday-button-buy-now" target="_blank">' . __( 'Get Your Deal', CHART_BUILDER_NAME ) . '</a>';
                        $content[] = '</div>';
                    $content[] = '</div>';
                $content[] = '</div>';

                $content[] = '<div style="position: absolute;right: 0;bottom: 1px;"  class="ays-chart-dismiss-buttons-container-for-form-black-friday">';
                    $content[] = '<form action="" method="POST">';
                        $content[] = '<div id="ays-chart-dismiss-buttons-content-black-friday">';
                            $content[] = '<button class="btn btn-link ays-button-black-friday" name="ays_chart_black_fr_sale_btn" style="">' . __( 'Dismiss ad', CHART_BUILDER_NAME ) . '</button>';
                        $content[] = '</div>';
                    $content[] = '</form>';
                $content[] = '</div>';
            $content[] = '</div>';

            $content = implode( '', $content );

            echo $content;
        }
    }

    // Christmas banner
    public static function ays_chart_christmas_message($ishmar){
        if($ishmar == 0 ){
            $content = array();

            $content[] = '<div id="ays-chart-dicount-christmas-month-main" class="notice notice-success is-dismissible ays_chart_dicount_info">';
                $content[] = '<div id="ays-chart-dicount-christmas-month" class="ays_chart_dicount_month">';
                    $content[] = '<div class="ays-chart-dicount-christmas-box">';
                        $content[] = '<div class="ays-chart-dicount-christmas-wrap-box ays-chart-dicount-christmas-wrap-box-80">';
                            $content[] = '<div class="ays-chart-dicount-christmas-title-row">' . __( 'Limited Time', CHART_BUILDER_NAME ) .' '. '<a href="https://ays-pro.com/wordpress/chart-builder" class="ays-chart-dicount-christmas-button-sale" target="_blank">' . __( '40%', CHART_BUILDER_NAME ) . '</a>' . ' SALE</div>';
                            $content[] = '<div class="ays-chart-dicount-christmas-title-row">' . __( 'Chart Builder Plugin', CHART_BUILDER_NAME ) . '</div>';
                        $content[] = '</div>';

                        $content[] = '<div class="ays-chart-dicount-christmas-wrap-box" style="width: 25%;">';
                            $content[] = '<div id="ays-chart-builder-countdown-main-container">';
                                $content[] = '<div class="ays-chart-countdown-container">';
                                    $content[] = '<div id="ays-chart-countdown" style="display: block;">';
                                        $content[] = '<ul>';
                                            $content[] = '<li><span id="ays-chart-countdown-days"></span>' . __( 'Days', CHART_BUILDER_NAME ) . '</li>';
                                            $content[] = '<li><span id="ays-chart-countdown-hours"></span>' . __( 'Hours', CHART_BUILDER_NAME ) . '</li>';
                                            $content[] = '<li><span id="ays-chart-countdown-minutes"></span>' . __( 'Minutes', CHART_BUILDER_NAME ) . '</li>';
                                            $content[] = '<li><span id="ays-chart-countdown-seconds"></span>' . __( 'Seconds', CHART_BUILDER_NAME ) . '</li>';
                                        $content[] = '</ul>';
                                    $content[] = '</div>';
                                    $content[] = '<div id="ays-chart-countdown-content" class="emoji" style="display: none;">';
                                        $content[] = '<span>🚀</span>';
                                        $content[] = '<span>⌛</span>';
                                        $content[] = '<span>🔥</span>';
                                        $content[] = '<span>💣</span>';
                                    $content[] = '</div>';
                                $content[] = '</div>';
                            $content[] = '</div>';
                        $content[] = '</div>';

                        $content[] = '<div class="ays-chart-dicount-christmas-wrap-box" style="width: 25%;">';
                            $content[] = '<a href="https://ays-pro.com/wordpress/chart-builder" class="ays-chart-dicount-christmas-button-buy-now" target="_blank">' . __( 'BUY NOW!', CHART_BUILDER_NAME ) . '</a>';
                        $content[] = '</div>';
                    $content[] = '</div>';
                $content[] = '</div>';

                $content[] = '<div style="position: absolute;right: 0;bottom: 1px;"  class="ays-chart-dismiss-buttons-container-for-form-christmas">';
                    $content[] = '<form action="" method="POST">';
                        $content[] = '<div id="ays-chart-dismiss-buttons-content-christmas">';
                            $content[] = '<button class="btn btn-link ays-button-christmas" name="ays_chart_sale_btn" style="">' . __( 'Dismiss ad', CHART_BUILDER_NAME ) . '</button>';
                        $content[] = '</div>';
                    $content[] = '</form>';
                $content[] = '</div>';
            $content[] = '</div>';

            $content = implode( '', $content );

            echo $content;
        }
    }

    // Silver Bundle
    public function ays_chart_silver_bundle_message($ishmar){
        if($ishmar == 0 ){
            $content = array();

            $content[] = '<div id="ays-chart-dicount-month-main" class="notice notice-success is-dismissible ays_chart_dicount_info">';
                $content[] = '<div id="ays-chart-dicount-month" class="ays_chart_dicount_month">';
                    $content[] = '<a href="https://ays-pro.com/silver-bundle?utm_source=dashboard-silver&utm_medium=free-silver&utm_campaign=chartify-silver" target="_blank" class="ays-chart-sale-banner-link"><img src="' . CHART_BUILDER_ADMIN_URL . '/images/silver_bundle_logo_box.png"></a>';

                    $content[] = '<div class="ays-chart-dicount-wrap-box">';

                        $content[] = '<strong style="font-weight: bold;">';
                            $content[] = __( "Limited Time <span style='color:#E85011;'>50%</span> SALE on <br><span><a href='https://ays-pro.com/silver-bundle?utm_source=dashboard-silver&utm_medium=free-silver&utm_campaign=chartify-silver' target='_blank' style='color:#E85011; text-decoration: underline;'>Silver Bundle</a></span> (Quiz + Chart + Form)!", CHART_BUILDER_NAME );
                        $content[] = '</strong>';

                        $content[] = '<br>';

                        $content[] = '<strong>';
                                $content[] = __( "Hurry up! <a href='https://ays-pro.com/silver-bundle?utm_source=dashboard-silver&utm_medium=free-silver&utm_campaign=chartify-silver' target='_blank'>Check it out!</a>", CHART_BUILDER_NAME );
                        $content[] = '</strong>';

                        $content[] = '<div style="position: absolute;right: 10px;bottom: 1px;" class="ays-chart-dismiss-buttons-container-for-form">';

                            $content[] = '<form action="" method="POST">';
                                $content[] = '<div id="ays-chart-dismiss-buttons-content">';
                                    $content[] = '<button class="btn btn-link ays-button" name="ays_chart_sale_btn" style="height: 32px; margin-left: 0;padding-left: 0">Dismiss ad</button>';
                                $content[] = '</div>';
                            $content[] = '</form>';
                            
                        $content[] = '</div>';

                    $content[] = '</div>';

                    $content[] = '<div class="ays-chart-dicount-wrap-box">';

                        $content[] = '<div id="ays-chart-builder-countdown-main-container">';
                            $content[] = '<div class="ays-chart-builder-countdown-container">';

                                $content[] = '<div id="ays-chart-countdown">';
                                    $content[] = '<ul>';
                                        $content[] = '<li><span id="ays-chart-countdown-days"></span>days</li>';
                                        $content[] = '<li><span id="ays-chart-countdown-hours"></span>Hours</li>';
                                        $content[] = '<li><span id="ays-chart-countdown-minutes"></span>Minutes</li>';
                                        $content[] = '<li><span id="ays-chart-countdown-seconds"></span>Seconds</li>';
                                    $content[] = '</ul>';
                                $content[] = '</div>';

                                $content[] = '<div id="ays-chart-countdown-content" class="emoji">';
                                    $content[] = '<span>🚀</span>';
                                    $content[] = '<span>⌛</span>';
                                    $content[] = '<span>🔥</span>';
                                    $content[] = '<span>💣</span>';
                                $content[] = '</div>';

                            $content[] = '</div>';
                        $content[] = '</div>';
                            
                    $content[] = '</div>';

                    $content[] = '<a href="https://ays-pro.com/silver-bundle?utm_source=dashboard-silver&utm_medium=free-silver&utm_campaign=chartify-silver" class="button button-primary ays-button" id="ays-button-top-buy-now" target="_blank" style="height: 32px; display: flex; align-items: center; font-weight: 500; " >' . __( 'Buy Now !', CHART_BUILDER_NAME ) . '</a>';
                $content[] = '</div>';
            $content[] = '</div>';

            $content = implode( '', $content );
            echo $content;
        }
    }

    public function ays_chart_update_banner_time(){

        $date = time() + ( 3 * 24 * 60 * 60 ) + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS);
        // $date = time() + ( 60 ) + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS); // for testing | 1 min
        $next_3_days = date('M d, Y H:i:s', $date);

        $ays_chart_banner_time = get_option('ays_chart_silver_bundle_banner_time');

        if ( !$ays_chart_banner_time || is_null( $ays_chart_banner_time ) ) {
            update_option('ays_chart_silver_bundle_banner_time', $next_3_days ); 
        }

        $get_ays_chart_banner_time = get_option('ays_chart_silver_bundle_banner_time');

        $val = 60*60*24*0.5; // half day
        // $val = 60; // for testing | 1 min

        $current_date = current_time( 'mysql' );
        $date_diff = strtotime($current_date) - intval(strtotime($get_ays_chart_banner_time));

        $days_diff = $date_diff / $val;
        if(intval($days_diff) > 0 ){
            update_option('ays_chart_silver_bundle_banner_time', $next_3_days);
        }

        return $get_ays_chart_banner_time;
    }

    public function author_user_search() {
        check_ajax_referer( 'cbuilder-author-user-search', 'security' );
        $params = $_REQUEST['params'];
        $search = isset($params['search']) && $params['search'] != '' ? sanitize_text_field( $params['search'] ) : null;
        $checked = isset($params['val']) && $params['val'] !='' ? sanitize_text_field( $params['val'] ) : null;
        $args = 'search=';
        if ($search !== null) {
            $args .= '*';
            $args .= $search;
            $args .= '*';
        }

        $users = get_users($args);

        $content_text = array(
            'results' => array()
        );

        foreach ($users as $key => $value) {
            if ($checked !== null) {
                if ( !is_array( $checked ) ) {
                    $checked2 = $checked;
                    $checked = array();
                    $checked[] = absint($checked2);
                }
                if (in_array($value->ID, $checked)) {
                    continue;
                } else {
                    $content_text['results'][] = array(
                        'id' => $value->ID,
                        'text' => $value->data->display_name,
                    );
                }
            } else {
                $content_text['results'][] = array(
                    'id' => $value->ID,
                    'text' => $value->data->display_name,
                );
            }
        }

        ob_end_clean();
        echo json_encode($content_text);
        wp_die();
    }

    /**
	 * Returns the data from Quiz maker.
	 *
	 * @access public
	 */
	public function fetch_quiz_maker_data(){
		check_ajax_referer( 'cbuilder-fetch-quiz-maker-data', 'security' );

		$params = $_POST['params'];
		if ( !isset($params) || empty($params) ) {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( "Something went wrong.", "chart-builder" )
				)
			);
		}
		
		$query = isset($params['query']) && $params['query'] !== '' ? $params['query'] : '';
		if ( $query == '' ) {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( "No data given.", "chart-builder" )
				)
			);
		}
		$quiz_id = isset($params['quizID']) && $params['quizID'] !== '' ? $params['quizID'] : null;
		$user_id = get_current_user_id();

		$return_results = CBFunctions()->get_quiz_query($query, $quiz_id, $user_id);

		if (empty($return_results)) {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( "Something went wrong.", "chart-builder" )
				)
			);
		}

		$result_values = $return_results['result'];
		$query = $return_results['query'];

        $results = array();
		if ( !empty($result_values) ) {
			$headers = array();
			if ( $result_values ) {
				$row_num = 0;
				foreach ( $result_values as $row ) {
					$result = array();
					$col_num = 0;
					foreach ( $row as $k => $v ) {
						$result[] = $v;
						if ( $row_num === 0 ) {
							$headers[]  = $k;
						}
					}
					$results[] = $result;
					$row_num++;
				}
			}
		}

		if (empty($results)) {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( "There is no data.", "chart-builder" )
				)
			);
		}

		foreach ($results as $key => $value) {
			if (!isset($value) || count($value) <= 1 && $key != 0) {
				unset($results[$key]);
			}
		}

		array_unshift($results, $headers);
		$results = array_values( $results );
		$data = $results;
		
		$headers = $results[0];
		unset( $results[0] );
		
		$html = CBFunctions()->get_table_html( $headers, $results );

		return array(
			'success' => true,
			'data' => array(
				'table' => $html,
				'data' => $data
			)
		);
	}

	/**
	 * Saves the Quiz maker data.
	 *
	 * @access public
	 */
	public function save_quiz_maker_data() {
		check_ajax_referer( 'cbuilder-save-quiz-maker-data', 'security' );

		$params = $_POST['params'];
		if ( !isset($params) || empty($params) ) {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( "Something went wrong.", "chart-builder" )
				)
			);
		}

		$query_id = isset($params['query']) && $params['query'] !== '' ? $params['query'] : '';
		if ( $query_id == '' ) {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( "No data given.", "chart-builder" )
				)
			);
		}

		$chart_id = isset( $params['chart_id'] ) ? absint( $params['chart_id'] ) : 0;
		$quiz_id = isset($params['quizID']) && $params['quizID'] !== '' ? $params['quizID'] : 0;
		$user_id = get_current_user_id();

		if ( $chart_id >= 0 ) {

			$return_results = CBFunctions()->get_quiz_query($query_id, $quiz_id, $user_id);

			if (empty($return_results)) {
				return array(
					'success' => false,
					'data' => array(
						'msg' => __( "Something went wrong.", "chart-builder" )
					)
				);
			}

			$result_values = $return_results['result'];
			$query = $return_results['query'];

            $results = array();
			if ( !empty($result_values) ) {
				$headers = array();
				if ( $result_values ) {
					$row_num = 0;
					foreach ( $result_values as $row ) {
						$result = array();
						$col_num = 0;
						foreach ( $row as $k => $v ) {
							$result[] = $v;
							if ( $row_num === 0 ) {
								$headers[]  = $k;
							}
						}
						$results[] = $result;
						$row_num++;
					}
				}
			}

            if (empty($results)) {
                return array(
                    'success' => false,
                    'data' => array(
                        'msg' => __( "There is no data.", "chart-builder" )
                    )
                );
            }

            $source_type = 'quiz_maker';
            $option_name_for_data = $chart_id == 0 ? 'ays_chart_quiz_maker_results_temp' : 'ays_chart_quiz_maker_results_' . $chart_id;
            $option_name_for_quiz = $chart_id == 0 ? 'ays_chart_quiz_maker_quiz_data_temp' : 'ays_chart_quiz_maker_quiz_data_' . $chart_id;

            update_option( $option_name_for_data, array(
                'source_type' => $source_type,
                'source' => $query,
                'data' => $results,
            ) );
            update_option( $option_name_for_quiz, array(
                'quiz_query' => $query_id,
                'quiz_id' => $quiz_id
            ) );

		} else {
			return array(
				'success' => false,
				'data' => array(
					'msg' => __( 'Given incorrect Chart ID.', "chart-builder")
				)
			);
		}

		return array(
			'success' => true,
			'data' => array(
				'msg' => __( 'Data was successfully saved.', "chart-builder" )
			)
		);
	}

	/**
	 * Chart page action hooks
	 */

	/**
     * Chart page sources contents
	 * @param $args
	 */
	public function ays_chart_page_source_contents( $args ){

		$sources_contents = apply_filters( 'ays_cb_chart_page_sources_contents_settings', array(), $args );

		$source_type = $args['source_type'];

		$sources = array();
		foreach ( $sources_contents as $key => $sources_content ) {
            $collapsed = $key == $source_type ? 'false' : 'true';

			$content = '<fieldset class="ays-accordion-options-container" data-collapsed="'. $collapsed .'">';
			if( isset( $sources_content['title'] ) ){
				$content .= '<legend class="ays-accordion-options-header">';
				$content .= '<svg class="ays-accordion-arrow '. ( $key == $source_type ? 'ays-accordion-arrow-down' : 'ays-accordion-arrow-right' ) .'" version="1.2" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" overflow="visible" preserveAspectRatio="none" viewBox="0 0 24 24" width="20" height="20">
                    <g>
                        <path xmlns:default="http://www.w3.org/2000/svg" d="M8.59 16.34l4.58-4.59-4.58-4.59L10 5.75l6 6-6 6z" style="fill: rgb(0, 140, 255);" vector-effect="non-scaling-stroke" />
                    </g>
                </svg>';

				$content .= '<span>'. $sources_content['title'] .'</span></legend>';
			}

			$content .= '<div class="ays-accordion-options-content">';
			$content .= $sources_content['content'];
			$content .= '</div>';

			$content .= '</fieldset>';

			$sources[] = $content;
		}
		$content_for_escape = implode('' , $sources );
		echo html_entity_decode(esc_html( $content_for_escape ));
	}

    public function source_contents_import_from_csv_settings( $sources, $args ){
	    $html_class_prefix = $args['html_class_prefix'];
	    $html_name_prefix = $args['html_name_prefix'];

        ob_start();
	    ?>
        <div class="ays-accordion-data-main-wrap only_pro">
            <div class="pro_features">
                <div>
                    <p>
                        <?php echo __("This feature is available only in ", "chart-builder"); ?>
                        <a href="https://ays-pro.com/wordpress/chart-builder/" target="_blank" title="PRO feature"><?php echo __("PRO version!!!", "chart-builder"); ?></a>
                    </p>
                </div>
            </div>
			<div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-main">
                <div class="<?php echo esc_attr($html_class_prefix) ?>file-import-form">
					<h6><?php echo __("Choose what kind of data would you like to upload.", "chart-builder"); ?></h6>
					<select class="form-select" style="max-width: none;" id="<?php echo esc_attr($html_class_prefix) ?>import-files-file-type">
						<option value="csv">CSV File</option>
					</select>
					<span style="display: block; font-style: italic; color: gray; font-size: 12px; margin: 5px 0;" class="<?php echo esc_attr($html_class_prefix) ?>small-hint-text"><?php echo __("Choose the format of the file you are going to import.", "chart-builder"); ?></span>
					<p class="<?php echo esc_attr($html_class_prefix) ?>csv-export-example" style="font-size: 15px; font-style: italic;">Example: 
						<a class="<?php echo esc_attr($html_class_prefix) ?>csv-export-example-link <?php echo esc_attr($html_class_prefix) ?>csv-export-example-link-other-types" style="cursor: pointer;">example.csv</a>
					</p>
					<div>
						<input style="padding: 0.5rem 1rem;" class="form-control form-control-md" id="<?php echo esc_attr($html_class_prefix) ?>import-files-input" name="<?php echo esc_attr($html_class_prefix) ?>import-files-input" type="file" accept=".csv"/>
					</div>
					<div class='ays-chart-file-import-success'></div>
					<div class='ays-chart-file-import-error'></div>
					<div class="ays-chart-buttons-group">
						<input type="button" class="button button-primary" id="ays-chart-file-import-fetch" value="<?php _e( "Show Results", "chart-builder" ); ?>">
            		    <input type="button" class="button button-primary" id="ays-chart-file-import-show-on-chart" value="<?php  _e( "Show on chart", "chart-builder" ); ?>">
            		    <input type="button" class="button button-primary" id="ays-chart-file-import-save" value="<?php  _e( "Save data", "chart-builder" ); ?>">
					</div>
				</div>
			</div>
        </div>
	    <?php
        $content = ob_get_clean();

	    $title = __( 'Import data from file', "chart-builder" ) . ' <a class="ays_help" data-bs-toggle="tooltip" title="' . __("With the help of this option, you can import a file in your chosen file format.","chart-builder") . '">
						<i class="ays_fa ays_fa_info_circle"></i>
					</a>';

	    $sources['file_import'] = array(
		    'content' => $content,
		    'title' => $title
	    );

        return $sources;
    }

	public function source_contents_import_from_db_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap only_pro">
			<div class="pro_features">
                <div>
                    <p>
                        <?php echo __("This feature is available only in ", $this->plugin_name); ?>
                        <a href="https://ays-pro.com/wordpress/chart-builder/" target="_blank" title="PRO feature"><?php echo __("PRO version!!!", $this->plugin_name); ?></a>
                    </p>
                </div>
            </div>
            <div class="<?= $html_class_prefix ?>source-data-main-wrap">
                <div class="<?= $html_class_prefix ?>chart-source-data-main">
                    <div id="ays-chart-db-query">
                        <div class="<?= $html_class_prefix ?>-db-query-form">
                            <div id="db-query-form">
                                <input type="hidden" name="chart_id" value="1">
                                <textarea name="query" class="<?= $html_class_prefix ?>db-query" placeholder="<?php _e( "Add your query here.", $this->plugin_name ); ?>"></textarea>
                                <div class='db-wizard-success'></div>
                                <div class='db-wizard-error'></div>
                            </div>
                            <div class="ays-chart-db-query-form-button">
                                <input type="button" class="button button-primary" id="ays-chart-query-fetch" value="<?php _e( "Show Results", $this->plugin_name ); ?>">
                                <input type="button" class="button button-primary" id="ays-chart-query-show-on-chart" value="<?php _e( "Show on chart", $this->plugin_name ); ?>">
                                <input type="button" class="button button-primary" id="ays-chart-query-save" value="<?php _e( "Save query", $this->plugin_name ); ?>">
                            </div>
                        </div>
                        <div class="db-wizard-hints">
                            <ul>
                                <!-- <li><//?php echo sprintf( __( 'For examples of queries and links to resources that you can use with this feature, please click %1$shere%2$s', $this->plugin_name ), '<a href="' . '#' . '" target="_blank">', '</a>' ); ?></li> -->
                                <li><?php echo sprintf( __( 'Use %1$sControl+Space%2$s for autocompleting keywords or table names.', $this->plugin_name ), '<span class="ays-chart-emboss">', '</span>' ); ?></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Connect to Database', $this->plugin_name ) . ' <a class="ays_help" data-bs-toggle="tooltip" title="' . __("Insert the Database query and the appropriate information from your Database will be displayed in the chart.",$this->plugin_name) . '">
					<i class="ays_fa ays_fa_info_circle"></i>
				</a>';

		$sources['import_from_db'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}

    public function source_contents_quiz_maker_integration_settings( $sources, $args ){
		global $wpdb;
		$chart_id = $args['chart_id'];
	    $html_class_prefix = $args['html_class_prefix'];
	    $html_name_prefix = $args['html_name_prefix'];
        $source = $args['source'];
        $settings = $args['settings'];
        $quiz_queries = $args['quiz_queries'];
		$quiz_id = $args['quiz_id'];
		$quiz_query = $args['quiz_query'];

		if ( !is_plugin_active('quiz-maker/quiz-maker.php') ) {
			// $title = sprintf( __( 'Get Quiz Maker data', "chart-builder" ) . ' <a class="ays_help" data-bs-toggle="tooltip" data-bs-html="true" title="' . __("By using this option, you can display the quiz statistics by charts.%s %sNote:%s The Quiz Maker plugin must be active.","chart-builder") . '">
			// 			<i class="ays_fa ays_fa_info_circle"></i>
			// 		</a>', '<br>', '<b>', '</b>');
			// $content = $this->blockquote_content_quiz;
	    	// $sources['quiz_maker'] = array(
			//     'content' => $content,
			//     'title' => $title
			// );

			return $sources;
		}

		$sql = "SELECT `title`, `id` FROM " . $wpdb->prefix . "aysquiz_quizes WHERE `published` = 1 AND `question_ids` <> ''";
		$quizes = $wpdb->get_results($sql, "ARRAY_A");

        ob_start();
	    ?>
        <div class="ays-accordion-data-main-wrap">
			<div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-main">
                <div id="<?php echo esc_attr($html_class_prefix) ?>quiz-maker-form">
					<input type="hidden" class="<?php echo esc_attr($html_class_prefix) ?>chart-id" value="<?= $chart_id ?>">
					<div class="<?php echo esc_attr($html_class_prefix) ?>select-quiz-maker-data-query-container">
						<select class="form-select" style="max-width: none;" id="<?php echo esc_attr($html_class_prefix) ?>select-quiz-maker-data-query" name="<?= $html_name_prefix ?>quiz_query">
							<option value=""><?= __( "Select query", "chart-builder" ) ?></option>
							<?php foreach ( $quiz_queries as $id => $q): 
                                $disabled = preg_replace('/[^0-9]/', '', $id) <= 5 ? "false" : "true" ; ?>
								<option value="<?= $id ?>" <?= $quiz_query == $id ? 'selected' : '' ?> <?php echo 'is-pro="'.$disabled.'"'; ?>><?= $q ?></option>
							<?php endforeach; ?>
						</select>
					</div>
					<div class="<?php echo esc_attr($html_class_prefix) ?>select-quiz-maker-quiz-container">
						<select class="form-select" style="max-width: none;" id="<?php echo esc_attr($html_class_prefix) ?>select-quiz-maker-quiz" name="<?= $html_name_prefix ?>quiz_id">
							<option value="0"><?= __( "Select quiz", "chart-builder" ) ?></option>
							<?php foreach ( $quizes as $quiz): ?>
								<option value="<?= $quiz['id'] ?>" <?= $quiz_id == $quiz['id'] ? 'selected' : '' ?> ><?= $quiz['title'] ?></option>
							<?php endforeach; ?>
						</select>
					</div>
					<div id="ays-chart-quiz-maker-success"></div>
					<div id="ays-chart-quiz-maker-error"></div>
					<div class="ays-chart-buttons-group">
						<input type="button" class="button button-primary" id="ays-chart-quiz-maker-fetch" value="<?php _e( "Show Results", "chart-builder" ); ?>">
            		    <input type="button" class="button button-primary" id="ays-chart-quiz-maker-show-on-chart" value="<?php  _e( "Show on chart", "chart-builder" ); ?>">
            		    <input type="button" class="button button-primary" id="ays-chart-quiz-maker-save" value="<?php  _e( "Save data", "chart-builder" ); ?>">
					</div>
				</div>
			</div>
        </div>
	    <?php
        $content = ob_get_clean();

	    $title = sprintf( __( 'Get Quiz Maker data', $this->plugin_name ) . ' <a class="ays_help" data-bs-toggle="tooltip" data-bs-html="true" title="' . __("By using this option, you can display the quiz statistics by charts.%s %sNote:%s The Quiz Maker plugin must be active.",$this->plugin_name) . '">
					<i class="ays_fa ays_fa_info_circle"></i>
				</a>', '<br>', '<b>', '</b>');
	    $sources['quiz_maker'] = array(
		    'content' => $content,
		    'title' => $title
	    );

        return $sources;
    }

    public function source_contents_manual_settings( $sources, $args ){
	    $html_class_prefix = $args['html_class_prefix'];
	    $html_name_prefix = $args['html_name_prefix'];
        $source = $args['source'];
        $settings = $args['settings'];

        ob_start();
	    ?>
        <div class="ays-accordion-data-main-wrap">
            <div class="<?php echo esc_attr($html_class_prefix) ?>source-data-main-wrap">
                <div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-main">
                    <!-- <div class="<//?= $html_class_prefix ?>icons-box">
                        <img class="<//?= $html_class_prefix ?>add-new-row" src="<//?php echo CHART_BUILDER_ADMIN_URL; ?>/images/icons/add-circle-outline.svg">
                    </div> -->
					<div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-content-container">
                    	<div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-content">
                    	    <?php if(!empty($source)):
                    	        foreach($source as $source_id => $source_value):
                    	            if(!empty($source_value) ):
                                        if ($source_id == 0): ?>
                                            <div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-edit-block" data-source-id = "<?php echo esc_attr($source_id); ?>">
                                                <div class="ays-chart-empty-data-table-cell"></div>
                                                <?php foreach($source_value as $each_source_id => $each_source_value): ?>
                                                    <div class="<?= $html_class_prefix ?>chart-source-data-input-box <?= $html_class_prefix ?>chart-source-title-box">
                                                        <svg class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-block <?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-col" data-trigger="hover" data-bs-toggle="tooltip" title="Delete column" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512" width="10px">
															<path d="M310.6 150.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L160 210.7 54.6 105.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L114.7 256 9.4 361.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L160 301.3 265.4 406.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L205.3 256 310.6 150.6z" style="fill: #b8b8b8;"/>
														</svg>
                                                        <input type="text" class="ays-text-input form-control <?= $html_class_prefix ?>chart-source-title-input" name="<?php echo esc_attr($html_name_prefix); ?>chart_source_data[<?php echo esc_attr($source_id); ?>][]" value="<?php echo stripslashes(esc_attr($each_source_value)); ?>">
                                                    </div>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php else: ?>
                    	                    <div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-edit-block" data-source-id = "<?php echo esc_attr($source_id); ?>">
                    	                        <div class="<?php echo esc_attr($html_class_prefix) ?>icons-box <?php echo esc_attr($html_class_prefix) ?>icons-remove-box">
										    		<svg class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-block <?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-row" data-trigger="hover" data-bs-toggle="tooltip" title="Delete row" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512">
										    			<path d="M310.6 150.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L160 210.7 54.6 105.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L114.7 256 9.4 361.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L160 301.3 265.4 406.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L205.3 256 310.6 150.6z" style="fill: #b8b8b8;"/>
										    		</svg>
                    	                        </div>
                    	                        <?php foreach($source_value as $each_source_id => $each_source_value): ?>
										    		<?php if ($each_source_id == 0): ?>
                    	                            	<div class="<?= $html_class_prefix ?>chart-source-data-input-box <?= $html_class_prefix ?>chart-source-data-name-input-box">
                    	                            	    <input type="text" class="ays-text-input form-control" name="<?php echo esc_attr($html_name_prefix); ?>chart_source_data[<?php echo esc_attr($source_id); ?>][]" value="<?php echo htmlspecialchars(esc_attr($each_source_value)); ?>">
                    	                            	</div>
										    		<?php else: ?>
										    			<div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-input-box <?php echo esc_attr($html_class_prefix) ?>chart-source-data-input-number">
                    	                            	    <input type="number" class="ays-text-input form-control" name="<?php echo esc_attr($html_name_prefix); ?>chart_source_data[<?php echo esc_attr($source_id); ?>][]" value="<?php echo stripslashes(esc_attr($each_source_value)); ?>">
                    	                            	</div>
										    		<?php endif; ?>
                    	                        <?php endforeach; ?>
                    	                    </div>
                                        <?php endif; ?>
                    	            <?php endif; ?>
                    	        <?php endforeach; ?>
                    	    <?php else:?>
                                <div class="<?= $html_class_prefix ?>chart-source-data-edit-block" data-source-id = "<?php echo esc_attr($source_id); ?>">
									<div style="height: 63.11px; padding: 0 15px;"></div>
                    	            <div class="<?= $html_class_prefix ?>chart-source-data-input-box <?= $html_class_prefix ?>chart-source-title-box">
                                        <svg class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-block <?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-col" data-trigger="hover" data-bs-toggle="tooltip" title="Delete column" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512" width="10px">
											<path d="M310.6 150.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L160 210.7 54.6 105.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L114.7 256 9.4 361.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L160 301.3 265.4 406.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L205.3 256 310.6 150.6z" style="fill: #b8b8b8;"/>
										</svg>
                    	                <input type="text" class="ays-text-input form-control <?= $html_class_prefix ?>chart-source-title-input" name="<?php echo esc_attr($html_name_prefix); ?>chart_source_data[0][]">
                    	            </div>
								</div>
                    	        <div class = "<?php echo esc_attr($html_class_prefix) ?>chart-source-data-edit-block" data-source-id="1">
                    	            <div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-input-box">
                    	                <input type="text" class="ays-text-input form-control" name="<?php echo esc_attr($html_name_prefix); ?>chart_source_data[1][]" >
                    	            </div>
                    	            <div class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-input-box <?php echo esc_attr($html_class_prefix) ?>chart-source-data-input-number">
                    	                <input type="number" class="ays-text-input form-control" name="<?php echo esc_attr($html_name_prefix); ?>chart_source_data[1][]">
                    	            </div>
                    	            <div class="<?php echo esc_attr($html_class_prefix) ?>icons-box <?php echo esc_attr($html_class_prefix) ?>icons-remove-box" >
                                        <svg class="<?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-block <?php echo esc_attr($html_class_prefix) ?>chart-source-data-remove-row" data-trigger="hover" data-bs-toggle="tooltip" title="Delete row" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512">
											<path d="M310.6 150.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L160 210.7 54.6 105.4c-12.5-12.5-32.8-12.5-45.3 0s-12.5 32.8 0 45.3L114.7 256 9.4 361.4c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L160 301.3 265.4 406.6c12.5 12.5 32.8 12.5 45.3 0s12.5-32.8 0-45.3L205.3 256 310.6 150.6z" style="fill: #b8b8b8;"/>
										</svg>
                    	            </div>
                    	        </div>
                    	    <?php endif; ?>
                    	</div>
						<div class="<?php echo esc_attr($html_class_prefix) ?>icons-box <?php echo esc_attr($html_class_prefix) ?>add-new-column-box" style="display: none;">
							<img class="<?php echo esc_attr($html_class_prefix) ?>add-new-column" src="<?php echo esc_url(CHART_BUILDER_ADMIN_URL); ?>/images/icons/add-circle-outline.svg">
							Add column
						</div>
					</div>
                    <div class="<?php echo esc_attr($html_class_prefix) ?>icons-box <?php echo esc_attr($html_class_prefix) ?>add-new-row-box">
                        <img class="<?php echo esc_attr($html_class_prefix) ?>add-new-row" src="<?php echo esc_url(CHART_BUILDER_ADMIN_URL); ?>/images/icons/add-circle-outline.svg">
						Add row
                    </div>
                    <br>
                    <button class="<?php echo esc_attr($html_class_prefix) ?>show-on-chart-bttn button button-primary ays-button">Show on Chart</button>
                </div>
            </div>
        </div>
	    <?php
        $content = ob_get_clean();

	    $title = __( 'Manual data', "chart-builder" ) . ' <a class="ays_help" data-bs-toggle="tooltip" title="' . __("Add the data manually. By clicking on the Add Row button you will be able to add as many rows as you need. While choosing the Line Chart type you will be able to also add the columns.","chart-builder") . '">
						<i class="ays_fa ays_fa_info_circle"></i>
					</a>';

	    $sources['manual'] = array(
		    'content' => $content,
		    'title' => $title
	    );

        return $sources;
    }

	/**
	 * Chart page settings contents
	 * @param $args
	 */
	public function ays_chart_page_settings_contents( $args ){

		$sources_contents = apply_filters( 'ays_cb_chart_page_settings_contents_settings', array(), $args );

		$sources = array();
		foreach ( $sources_contents as $key => $sources_content ) {
			$collapsed = $key == 'general_settings' ? 'false' : 'true';

			$content = '<fieldset class="ays-accordion-options-container" data-collapsed="' . $collapsed . '">';
			if(isset($sources_content['title'])){
				$content .= '<legend class="ays-accordion-options-header">';
				$content .= '<svg class="ays-accordion-arrow ays-accordion-arrow-right" version="1.2" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" overflow="visible" preserveAspectRatio="none" viewBox="0 0 24 24" width="20" height="20">
                    <g>
                        <path xmlns:default="http://www.w3.org/2000/svg" d="M8.59 16.34l4.58-4.59-4.58-4.59L10 5.75l6 6-6 6z" style="fill: rgb(0, 140, 255);" vector-effect="non-scaling-stroke" />
                    </g>
                </svg>';

				$content .= '<span>'. esc_html($sources_content['title']) .'</span></legend>';
			}

			$content .= '<div class="ays-accordion-options-content">';
				$content .= $sources_content['content'];
			$content .= '</div>';

			$content .= '</fieldset>';

			$sources[] = $content;
		}
		$content_for_escape = implode('' , $sources );
		echo html_entity_decode(esc_html( $content_for_escape ));
	}

	public function settings_contents_general_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
        $chart_description = $args['chart_description'];
        $create_author_data = $args['create_author_data'];
		$status = $args['status'];
		$settings = $args['settings'];

        $show_title = $settings['show_title'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-status" class="form-label">
                            <?php echo esc_html(__('Chart status', "chart-builder")); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("Decide whether the chart is active or not. If the chart is a draft, it won't be shown anywhere on your website (you don't need to remove shortcodes).","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
						<label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                            <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-status" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>status" value="published" <?php echo $status == 'published' ? 'checked' : ''; ?> >
                            <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                        </label>
                    </div>
                </div> <!-- Status -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-description" class="form-label">
                            <?php echo esc_html(__( "Chart description", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("Set the chart description","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <textarea class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-textarea-input" id="ays-chart-description" type="text" name="<?php echo esc_attr($html_name_prefix); ?>description"><?php echo esc_attr($chart_description) ?></textarea>
                    </div>
                </div> <!-- Chart description -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-create-author" class="form-label">
                            <?php echo esc_html(__('Change the author of the current chart',"chart-builder")); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_html(__('You can change the author who created the current chart to your preferred one. You need to write the User ID here. Please note, that in case you write an ID, by which there are no users found, the changes will not be applied and the previous author will remain the same.',"chart-builder")); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-create-author" name="<?php echo esc_attr($html_name_prefix); ?>create_author">
                            <option value=""><?php echo esc_html(__('Select User',"chart-builder"))?></option>
				            <option value="<?php echo esc_html($create_author_data['ID'])?>" selected><?php echo esc_html($create_author_data['display_name'])?></option>
                        </select>
                    </div>
                </div> <!-- Change chart author -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-show-title" class="form-label">
                            <?php echo esc_html(__('Show chart title', "chart-builder")); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("If you disable the toggle, the Chart title will not be displayed on the Front-end. By default, the toggle is enabled.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
						<label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                            <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-show-title" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[show_title]" value="on" <?php echo esc_attr($show_title); ?> >
                            <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                        </label>
                    </div>
                </div> <!-- Show title -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'General Settings', "chart-builder" );

		$sources['general_settings'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}

	public function settings_contents_styles_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];

		$width = $settings['width'];
		$height = $settings['height'];
		$font_size = $settings['font_size'];
		$title_color = $settings['title_color'];
		$title_font_size = $settings['title_font_size'];
        $description_color = $settings['description_color'];
		$description_font_size = $settings['description_font_size'];
		$title_positions = $settings['title_positions'];
        $title_position = $settings['title_position'];
		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <h6><?php echo __("General", "chart-builder"); ?></h6>
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-width" class="form-label">
                            <?php echo esc_html(__( "Width", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-width" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[width]" value="<?php echo esc_attr($width) ?>">
						<div class="<?php echo esc_attr($html_class_prefix) ?>option-desc-box"><img src="<?php echo esc_url(CHART_BUILDER_ADMIN_URL); ?>/images/icons/percent.svg"></div>
                    </div>
                </div> <!-- Width -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-height" class="form-label">
                            <?php echo esc_html(__( "Height", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-height" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[height]" value="<?php echo esc_attr($height) ?>">
						<div class="<?php echo esc_attr($html_class_prefix) ?>option-desc-box">px</div>
                    </div>
                </div> <!-- Height -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-font-size" class="form-label">
                            <?php echo esc_html(__( "Font size", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-font-size" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[font_size]" value="<?php echo esc_attr($font_size) ?>">
						<div class="<?php echo esc_attr($html_class_prefix) ?>option-desc-box">px</div>
                    </div>
                </div> <!-- Font size -->
                <br>
                <h6><?php echo __("Title", "chart-builder"); ?></h6>
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-title-color">
				            <?php echo esc_html(__( "Chart title color", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input id="ays-chart-option-title-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[title_color]" value="<?php echo esc_attr($title_color) ?>">
                    </div>
                </div> <!-- Chart title color -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-title-font-size" class="form-label">
                            <?php echo esc_html(__( "Chart title font size", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-title-font-size" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[title_font_size]" value="<?php echo esc_attr($title_font_size) ?>">
						<div class="<?php echo esc_attr($html_class_prefix) ?>option-desc-box">px</div>
                    </div>
                </div> <!-- Chart title font size -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-title-position">
				            <?php echo esc_html(__( "Chart title position", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input"">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-title-position" name="<?php echo esc_attr($html_name_prefix); ?>settings[title_position]">
                            <?php
				            foreach ( $title_positions as $option_slug => $option ):
					            $selected = ( $title_position == $option_slug ) ? 'selected' : '';
					            ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Title position -->
                <br>
                <h6><?php echo __("Description", "chart-builder"); ?></h6>
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-description-color">
				            <?php echo esc_html(__( "Chart description color", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input id="ays-chart-option-description-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[description_color]" value="<?php echo esc_attr($description_color) ?>">
                    </div>
                </div> <!-- Chart description color -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-description-font-size" class="form-label">
                            <?php echo esc_html(__( "Chart description font size", "chart-builder" )); ?>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-description-font-size" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[description_font_size]" value="<?php echo esc_attr($description_font_size) ?>">
						<div class="<?php echo esc_attr($html_class_prefix) ?>option-desc-box">px</div>
                    </div>
                </div> <!-- Chart description font size -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Styles', "chart-builder" );

		$sources['styles'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}

	public function settings_contents_tooltip_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];

        $tooltip_trigger_options = $settings['tooltip_trigger_options'];
		$tooltip_trigger = $settings['tooltip_trigger'];
		$show_color_code = $settings['show_color_code'];
		$tooltip_text_color = $settings['tooltip_text_color'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-tooltip-trigger">
				            <?php echo esc_html(__( "Trigger", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Choose when to display the results on the chart.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-tooltip-trigger" name="<?php echo esc_attr($html_name_prefix); ?>settings[tooltip_trigger]">
				            <?php
				            foreach ( $tooltip_trigger_options as $option_slug => $option ):
					            $selected = ( $tooltip_trigger == $option_slug ) ? 'selected' : '';
					            ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Trigger -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-show-color-code">
				            <?php echo esc_html(__( "Show Color Code", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The color will be displayed while clicking on a particular part of the chart.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                            <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-option-show-color-code" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[show_color_code]" value="on" <?php echo esc_attr($show_color_code); ?> >
                            <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                        </label>
                    </div>
                </div> <!-- Show Color Code -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-tooltip-text-color">
				            <?php echo esc_html(__( "Text color", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The color of the tooltip text.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input id="ays-chart-option-tooltip-text-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[tooltip_text_color]" value="<?php echo esc_attr($tooltip_text_color) ?>">
                    </div>
                </div> <!-- Text color -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Tooltip', "chart-builder" );

		$sources['tooltip'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}

    public function settings_contents_legend_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];
        $legend_positions = $settings['legend_positions'];
        $legend_position = $settings['legend_position'];
        $legend_alignments = $settings['legend_alignments'];
        $legend_alignment = $settings['legend_alignment'];
        $legend_color = $settings['legend_color'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-legend-position">
				            <?php echo esc_html(__( "Position", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Choose the appropriate position for the chart legend. You will be able to also hide it by choosing the 'Omit the legend' option.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-legend-position" name="<?php echo esc_attr($html_name_prefix); ?>settings[legend_position]">
                            <?php
				            foreach ( $legend_positions as $option_slug => $option ):
					            $selected = ( $legend_position == $option_slug ) ? 'selected' : '';
					            ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Legend position -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-legend-alignment">
				            <?php echo esc_html(__( "Alignment", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Choose the appropriate alignment for the chart legend.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-legend-alignment" name="<?php echo esc_attr($html_name_prefix); ?>settings[legend_alignment]">
                            <?php
				            foreach ( $legend_alignments as $option_slug => $option ):
					            $selected = ( $legend_alignment == $option_slug ) ? 'selected' : '';
					            ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Legend alignment -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-legend-font-color">
				            <?php echo esc_html(__( "Font Color", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Choose the font color for the chart legend.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input id="ays-chart-option-legend-font-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[legend_color]" value="<?php echo esc_attr($legend_color) ?>">
                    </div>
                </div> <!-- Legend font color -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Legend', "chart-builder" );

		$sources['legend'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}

    public function settings_contents_horizontal_axis_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];

        $haxis_title = $settings['haxis_title'];
        $axes_text_positions = $settings['axes_text_positions'];
        $haxis_text_position = $settings['haxis_text_position'];
        $haxis_direction = $settings['haxis_direction'];
        $haxis_text_color = $settings['haxis_text_color'];
        $haxis_slanted_options = $settings['haxis_slanted_options'];
        $haxis_slanted = $settings['haxis_slanted'];
        $haxis_slanted_text_angle = $settings['haxis_slanted_text_angle'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap <?= $html_class_prefix ?>options-haxis-settings-tab">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-haxis-title" class="form-label">
                            <?php echo esc_html(__( "Axis title", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("The title of the horizontal axis","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-haxis-title" type="text" name="<?php echo esc_attr($html_name_prefix); ?>settings[haxis_title]" value="<?php echo esc_attr($haxis_title) ?>">
                    </div>
                </div> <!-- Horizontal axis title -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-haxis-text-position" class="form-label">
                            <?php echo esc_html(__( "Text position", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Position of the horizontal axis text, relative to the chart area.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-haxis-text-position" name="<?php echo esc_attr($html_name_prefix); ?>settings[haxis_text_position]">
                            <?php
				            foreach ( $axes_text_positions as $option_slug => $option ):
				    	        $selected = ( $haxis_text_position == $option_slug ) ? 'selected' : '';
				    	        ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Horizontal axis text position -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-haxis-text-direction">
				            <?php echo esc_html(__( "Reverse Direction", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The direction in which the values grow along the horizontal axis. By default, low values are on the left of the chart.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                            <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-option-haxis-text-direction" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[haxis_direction]" value="-1" <?php echo esc_attr($haxis_direction); ?> >
                            <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                        </label>
                    </div>
                </div> <!-- Horizontal axis text direction -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-haxis-text-color">
				            <?php echo esc_html(__( "Axis text color", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The color of the horizontal axis text.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input id="ays-chart-option-haxis-text-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[haxis_text_color]" value="<?php echo esc_attr($haxis_text_color) ?>">
                    </div>
                </div> <!-- Horizontal axis text color -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-haxis-slanted-text" class="form-label">
                            <?php echo esc_html(__( "Slanted text", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("By choosing true the text will be slanted. In false it will be horizontal. In Automatic based on the size, it will be displayed either in horizontal or slanted. Note: This option only works, if 'Text position' option is set to 'Outside the chart'","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-haxis-slanted-text" name="<?php echo esc_attr($html_name_prefix); ?>settings[haxis_slanted]">
                            <?php
				            foreach ( $haxis_slanted_options as $option_slug => $option ):
				    	        $selected = ( $haxis_slanted == $option_slug ) ? 'selected' : '';
				    	        ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Horizontal axis slanted text -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section <?php echo ($haxis_slanted == 'false') ? 'display_none' : ''; ?>">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-haxis-slanted-text-angle" class="form-label">
                            <?php echo esc_html(__( "Slanted text angle", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The slanted text angle will define the angle. It will tilt in between -90 from 90 (except 0).","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-haxis-slanted-text-angle" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[haxis_slanted_text_angle]" value="<?php echo esc_attr($haxis_slanted_text_angle) ?>" step="15" min="-90" max="90">
                    </div>
                </div> <!-- Horizontal axis slanted text angle -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Horizontal Axis Settings', "chart-builder" );

		$sources['horizontal_axis'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}
    
    public function settings_contents_vertical_axis_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];

        $vaxis_title = $settings['vaxis_title'];
        $axes_text_positions = $settings['axes_text_positions'];
        $vaxis_text_position = $settings['vaxis_text_position'];
        $vaxis_direction = $settings['vaxis_direction'];
        $vaxis_text_color = $settings['vaxis_text_color'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap <?= $html_class_prefix ?>options-vaxis-settings-tab">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-vaxis-title" class="form-label">
                            <?php echo esc_html(__( "Axis title", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("The title of the vertical axis","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-vaxis-title" type="text" name="<?php echo esc_attr($html_name_prefix); ?>settings[vaxis_title]" value="<?php echo esc_attr($vaxis_title) ?>">
                    </div>
                </div> <!-- Vertical axis title -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-vaxis-text-position" class="form-label">
                            <?php echo esc_html(__( "Text position", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Position of the vertical axis text, relative to the chart area.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                        <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-vaxis-text-position" name="<?php echo esc_attr($html_name_prefix); ?>settings[vaxis_text_position]">
                            <?php
				            foreach ( $axes_text_positions as $option_slug => $option ):
				    	        $selected = ( $vaxis_text_position == $option_slug ) ? 'selected' : '';
				    	        ?>
                                <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				            <?php
				            endforeach;
				            ?>
                        </select>
                    </div>
                </div> <!-- Vertical text position -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-vaxis-text-direction">
				            <?php echo esc_html(__( "Reverse Direction", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The direction in which the values along the vertical axis grow. By default, low values are on the bottom of the chart.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                            <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-option-vaxis-text-direction" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[vaxis_direction]" value="-1" <?php echo esc_attr($vaxis_direction); ?> >
                            <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                        </label>
                    </div>
                </div> <!-- Vertical axis text direction -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-vaxis-text-color">
				            <?php echo esc_html(__( "Axis text color", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The color of the vertical axis text.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input id="ays-chart-option-vaxis-text-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[vaxis_text_color]" value="<?php echo esc_attr($vaxis_text_color) ?>">
                    </div>
                </div> <!-- Vertical axis text color -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Vertical Axis Settings', "chart-builder" );

		$sources['vertical_axis'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
	}

    public function settings_contents_animation_settings( $sources, $args ) {
        $html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];

        $enable_animation = $settings['enable_animation'];
        $animation_duration = $settings['animation_duration'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap <?= $html_class_prefix ?>options-vaxis-settings-tab">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-enable-animation">
				            <?php echo esc_html(__( "Chart animation", "chart-builder" )); ?>
							<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("Enable chart animation on window load.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                            <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-option-enable-animation" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[enable_animation]" value="on" <?php echo esc_attr($enable_animation); ?> >
                            <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                        </label>
                    </div>
                </div> <!-- Enable animation -->
                <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section <?php echo ($enable_animation == 'checked') ? '' : 'display_none'; ?>">
                    <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                        <label for="ays-chart-option-animation-duration" class="form-label">
                            <?php echo esc_html(__( "Animation duration", "chart-builder" )); ?>
                            <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The duration of the chart animation, in milliseconds.","chart-builder") ); ?>">
                                <i class="ays_fa ays_fa_info_circle"></i>
                            </a>
                        </label>
                    </div>
                    <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                        <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-animation-duration" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[animation_duration]" value="<?php echo esc_attr($animation_duration) ?>">
                    </div>
                </div> <!-- Animation duration -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Chart animation', "chart-builder" );

		$sources['animation'] = array(
			'content' => $content,
			'title' => $title
		);

		return $sources;
    }

    /**
	 * Chart page settings contents
	 * @param $args
	 */
	public function ays_chart_page_advanced_settings_contents( $args ){

		$sources_contents = apply_filters( 'ays_cb_chart_page_advanced_settings_contents_settings', array(), $args );

		$sources = array();
		foreach ( $sources_contents as $key => $sources_content ) {
			$collapsed = $key == 'advanced_settings' ? 'false' : 'true';

			$content = '<fieldset class="ays-accordion-options-container" data-collapsed="' . $collapsed . '">';
			if(isset($sources_content['title'])){
				$content .= '<legend class="ays-accordion-options-header">';
				$content .= '<svg class="ays-accordion-arrow ays-accordion-arrow-right" version="1.2" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" overflow="visible" preserveAspectRatio="none" viewBox="0 0 24 24" width="20" height="20">
                    <g>
                        <path xmlns:default="http://www.w3.org/2000/svg" d="M8.59 16.34l4.58-4.59-4.58-4.59L10 5.75l6 6-6 6z" style="fill: rgb(0, 140, 255);" vector-effect="non-scaling-stroke" />
                    </g>
                </svg>';

				$content .= '<span>'. esc_html($sources_content['title']) .'</span></legend>';
			}

			$content .= '<div class="ays-accordion-options-content">';
				$content .= $sources_content['content'];
			$content .= '</div>';

			$content .= '</fieldset>';

			$sources[] = $content;
		}
		$content_for_escape = implode('' , $sources );
		echo html_entity_decode(esc_html( $content_for_escape ));
	}

	public function settings_contents_advanced_settings( $sources, $args ){
		$html_class_prefix = $args['html_class_prefix'];
		$html_name_prefix = $args['html_name_prefix'];
		$settings = $args['settings'];
        $tab_title = $args['tab_name'];

        $rotation_degree = $settings['rotation_degree'];
        $reverse_categories = $settings['reverse_categories'];
        $slice_border_color = $settings['slice_border_color'];
        $slice_texts = $settings['slice_texts'];
        $slice_text = $settings['slice_text'];
        $tooltip_text_options = $settings['tooltip_text_options'];
		$tooltip_text = $settings['tooltip_text'];
        $is_stacked = $settings['is_stacked'];
        $focus_target_options = $settings['focus_target_options'];
		$focus_target = $settings['focus_target'];
		$opacity = $settings['opacity'];
		$line_width = $settings['line_width'];

		ob_start();
		?>
        <div class="ays-accordion-data-main-wrap">
            <div class="<?php echo esc_attr($html_class_prefix) ?>settings-data-main-wrap">
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section" style="display: none;">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-rotation-degree" class="form-label">
                                <?php echo esc_html(__( "Degree of chart rotation", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The angle, in degrees, to rotate the chart by. The default of 0 will orient the leftmost edge of the first slice directly up.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                            <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-rotation-degree" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[rotation_degree]" value="<?php echo esc_attr($rotation_degree) ?>">
                        </div>
                    </div> <!-- Rotation degree -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section" style="display: none;">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-reverse-categories">
                                <?php echo esc_html(__( "Reverse Categories", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("If this option is enabled, the pie slices will be drawn counterclockwise.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                            <label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                                <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-option-reverse-categories" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[reverse_categories]" value="on" <?php echo esc_attr($reverse_categories); ?> >
                                <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                            </label>
                        </div>
                    </div> <!-- Reverse Categories -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section" style="display: none;">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-slice-border-color">
				                <?php echo esc_html(__( "Slice Border Color", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The color of the slice borders.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                            <input id="ays-chart-option-slice-border-color" class="form-control-color <?php echo esc_attr($html_class_prefix) ?>option-color-picker" type="color" name="<?php echo esc_attr($html_name_prefix); ?>settings[slice_border_color]" value="<?php echo esc_attr($slice_border_color) ?>">
                        </div>
                    </div> <!-- Slice border color -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section" style="display: none;">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-slice-text">
				                <?php echo esc_html(__( "Slice text", "chart-builder" )); ?>
				    			<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Choose the content of the text to be displayed on the pie slice.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                            <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-slice-text" name="<?php echo esc_attr($html_name_prefix); ?>settings[slice_text]">
                                <?php
				                foreach ( $slice_texts as $option_slug => $option ):
				    	            $selected = ( $slice_text == $option_slug ) ? 'selected' : '';
				    	            ?>
                                    <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				                <?php
				                endforeach;
				                ?>
                            </select>
                        </div>
                    </div> <!-- Slice text -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section" style="display: none;">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-tooltip-text">
				                <?php echo esc_html(__( "Slice tooltip text", "chart-builder" )); ?>
				    			<a class="ays_help" data-bs-toggle="tooltip" title="<?php echo htmlspecialchars( __("Choose how to display the text in the chart tooltip.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                            <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-tooltip-text" name="<?php echo esc_attr($html_name_prefix); ?>settings[tooltip_text]">
				                <?php
				                foreach ( $tooltip_text_options as $option_slug => $option ):
				    	            $selected = ( $tooltip_text == $option_slug ) ? 'selected' : '';
				    	            ?>
                                    <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
				                <?php
				                endforeach;
				                ?>
                            </select>
                        </div>
                    </div> <!-- Slice tooltip Text -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-is-stacked">
                                <?php echo esc_html(__( "Is stacked", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("Is stacked: If this option is enabled, the chart will be displayed in the proportional contribution of individual data points in comparison to a total.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 py-2 <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                            <label class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-switch">
                                <input class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch" id="ays-chart-option-is-stacked" type="checkbox" name="<?php echo esc_attr($html_name_prefix); ?>settings[is_stacked]" value="on" <?php echo esc_attr($is_stacked); ?> >
                                <span class="<?php echo esc_attr($html_class_prefix) ?>toggle-switch-slider <?php echo esc_attr($html_class_prefix) ?>toggle-switch-round"></span>
                            </label>
                        </div>
                    </div> <!-- Is stacked -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-focus-target">
                                <?php echo esc_html(__( "Focus target", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" data-bs-html="true" title="<?php 
                                    echo htmlspecialchars( sprintf(
                                        "<p>" . __('There are two ways to focus the data table elements.', "chart-builder") . "</p><ul class='ays_tooltop_ul'><li>" .
                                        __('%sSingle data:%s If you choose the Single Data option, the focus will be on the single data point. By this, the particular cell of the data table will be focused.', "chart-builder") . "</li><li>" .
                                        __('%sGroup data:%s If you choose the Group Data option, the focus will be on the grouped data points. By this, a row of the data table will be focused.', "chart-builder") . "</li></ul>",
                                        '<em>',
                                        '</em>',
                                        '<em>',
                                        '</em>'
                                    ) );
                                ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input">
                            <select class="<?php echo esc_attr($html_class_prefix) ?>option-select-input form-select" id="ays-chart-option-focus-target" name="<?php echo esc_attr($html_name_prefix); ?>settings[focus_target]">
                                <?php
                                foreach ( $focus_target_options as $option_slug => $option ):
                                    $selected = ( $focus_target == $option_slug ) ? 'selected' : '';
                                    ?>
                                    <option value="<?php echo esc_attr($option_slug); ?>" <?php echo esc_attr($selected); ?>><?php echo esc_html($option); ?></option>
                                <?php
                                endforeach;
                                ?>
                            </select>
                        </div>
                    </div> <!-- Focus target -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-opacity" class="form-label">
                                <?php echo esc_html(__( "Opacity", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("The transparency of data points, with 1.0 being completely opaque and 0.0 fully transparent. In bar and column charts, this refers to the visible data: rectangles. In charts where selecting data creates a dot, such as the line chart, this refers to the circles that appear upon hover or selection.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                            <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-opacity" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[opacity]" value="<?php echo esc_attr($opacity) ?>" step=".1" min="0" max="1">
                        </div>
                    </div> <!-- Data opacity -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section" style="display: none;">
                        <div class="col-sm-5 d-flex align-items-center <?php echo esc_attr($html_class_prefix) ?>option-title">
                            <label for="ays-chart-option-line-width" class="form-label">
                                <?php echo esc_html(__( "Chart line width", "chart-builder" )); ?>
                                <a class="ays_help" data-bs-toggle="tooltip" title="<?php echo esc_attr( __("Data line width in pixels. Set 0 for hiding all the lines and show only the point.","chart-builder") ); ?>">
                                    <i class="ays_fa ays_fa_info_circle"></i>
                                </a>
                            </label>
                        </div>
                        <div class="col-sm-7 <?php echo esc_attr($html_class_prefix) ?>option-input <?php echo esc_attr($html_class_prefix) ?>input-align-right">
                            <input class="ays-text-input form-control <?php echo esc_attr($html_class_prefix) ?>option-text-input" id="ays-chart-option-line-width" type="number" name="<?php echo esc_attr($html_name_prefix); ?>settings[line_width]" value="<?php echo esc_attr($line_width) ?>">
                        </div>
                    </div> <!-- Line width -->
                    <div class="form-group row mb-2 <?php echo esc_attr($html_class_prefix) ?>options-section only_pro" style="display: none;">
                        <div class="pro_features">
                            <div>
                                <p>
                                    <?php echo __("This feature is available only in ", $this->plugin_name); ?>
                                    <a href="https://ays-pro.com/wordpress/chart-builder/" target="_blank" title="PRO feature"><?php echo __("PRO version!!!", $this->plugin_name); ?></a>
                                </p>
                            </div>
                        </div>
                        <div class="col-sm-5 d-flex align-items-center <?= $html_class_prefix ?>option-title">
                            <label for="ays-chart-option-curve-type">
                                <?php echo __( "Line curve type", "chart-builder" ); ?>
                            </label>
                        </div>
                        <div class="col-sm-7 <?= $html_class_prefix ?>input-align-right">
                            <select class="<?= $html_class_prefix ?>option-select-input form-select" id="ays-chart-option-line-curve-type" name="<?php echo $html_name_prefix; ?>settings[line_curve_type]">
                                <option><?php echo __( "Straight", "chart-builder" ); ?></option>
                            </select>
                        </div>
                    </div> <!-- Line curve type -->
            </div>
        </div>
		<?php
		$content = ob_get_clean();

		$title = __( 'Advanced Settings', "chart-builder" );

		$sources['advanced_settings'] = array(
			'content' => $content,
			'title' => $tab_title
		);

		return $sources;
	}
}
